/******************************************************************************
 CBkgndTask.h

			Source File For CBkgndTask Class
	

	Copyright (C) 1985-1992  New York University
	Copyright (C) 1994 George Washington University
	 
	This file is part of the GWAdaEd system, an extension of the Ada/Ed-C
	system.  See the Ada/Ed README file for warranty (none) and distribution
	info and also the GNU General Public License for more details.


	by Manuel A. Perez

 ******************************************************************************/

#include "LaunchApp.h"
#include "CBkgndTask.h"
#include "CAdaApp.h"
#include "CLibrary.h"
#include "CFileMgr.h"
#include "AdaGlobals.h"

#include <TBUtilities.h>
#include <Exceptions.h>
#include <stdio.h>
#include <stdlib.h>
#include "ErrMessages.h"
#include "Announce.h"


/****
 *	IBkgndTask
 *
 ****/

void	CBkgndTask::IBkgndTask(Str255 name)
{
Str255 path;

	next = NULL;
	errorCode = RC_SUCCESS;

	gFileMgr->GetCompilerPath(path);
	CopyPString(path, procPath);
	CopyPString(name, procName);
	CopyPString(procPath, procFullName);
	ConcatPStrings( procFullName, procName);

	// Don't call this here!  Call it when the compiler part is about
	// to start executing.  This routine is called for two compiler
	// parts (adafront and adagen)  before either one is executing.
	// gFileMgr->SetPart(procName);
}

/****
 *	Dispose()
 *
 ****/

void	CBkgndTask::Dispose(void)
{
	next = NULL;
	procPath[0] = 0;
	procName[0] = 0;
	procFullName[0] = 0;

	inherited::Dispose();
}

/****
 *	SetPartID
 *
 ****/

void	CBkgndTask:: SetPartID(CompilerPart part)
{
	partId = part;
}

/****
 *	PartID
 *
 ****/

CompilerPart	CBkgndTask:: PartID(void)
{
	return partId;
}

/****
 *	SetNext()
 *
 ****/

void	CBkgndTask::SetNext(CBkgndTask *p)
{
	next = p;
}


/****
 *	GetNext()
 *
 ****/

CBkgndTask *CBkgndTask::GetNext(void)
{
	return next;
}

/****
 *	StartRunning()
 *
 ****/

void	CBkgndTask::StartRunning(void)
{
	OSErr err;

	gFileMgr->SetPart(procName);
	err = FSMakeFSSpec(0, 0, procFullName, &bkgndPgm);
	if (err == fnfErr)
		Failure(errCompilerNotFound, 0);

	WriteOptions();


	err = LaunchAnApplication(bkgndPgm, true, &procNum);
	if (err == memFullErr || err == memFragErr)
		Failure(errNotEnoughMemory, 0);

	else if (err != noErr)
		Failure(errInternalError, 0);
}

/****
 *	WriteOptions()
 *
 ****/

void	CBkgndTask::WriteOptions(void)
{
}

/****
 *	StillRunning()
 *
 ****/

Boolean	CBkgndTask::StillRunning(void)
{
	return StillGoing(procNum);
}

/****
 *	ReadCompletionCode
 *
 *	Call this routine only once, right after the background process
 *	finished running.
 *
 ****/

int		CBkgndTask:: ReadCompletionCode(void)
{
	Str255 returnFile;
	FILE *fp;
	
	/* we just finished, lets get any possible error codes
	 *
	 * Return codes from the running program are stored in
	 * file with extension ".return".
	 */

	gFileMgr->GetReturnFile(returnFile);	

	/* now we have the full name, change it to a C string
	 * and try to open the file
	 */
	
	PtoCstr(returnFile);
	fp = fopen((char *)returnFile, "r");
	if (fp == NULL) {
		/* no errors were found, do whatever is needed */
		errorCode = RC_SUCCESS;
	}
	else {
		char str[128];
		int i;
		
		fgets(str, 128, fp);
		switch (i = atoi(str)) {
			case RC_SUCCESS:
			case RC_ERRORS:
				errorCode = i;
				break;
			case RC_ABORT:		// what to do with this case?
			case RC_INTERNAL_ERROR:
			default:
				errorCode = RC_INTERNAL_ERROR;
				break;
		}
		fclose(fp);
	}

	remove((char *)returnFile);	// delete the error file

	return errorCode;
}

/****
 *	CompletionCode
 *
 *	Return the error code.  Must call ReadCompletionCode first.
 *
 ****/

int		CBkgndTask::CompletionCode(void)
{
	return errorCode;
}

/****
 *	AnnounceWithSound()
 *
 *	Play a sound according to the error code returned by the
 *	background process.
 *
 ****/

void	CBkgndTask::AnnounceWithSound(void)
{
	switch (CompletionCode()) {
		case RC_INTERNAL_ERROR:
			Announce("\pInternal Error");
			break;
		case RC_ERRORS:
			Announce("\pCompile Error");
			break;
		case RC_SUCCESS:
			Announce("\pCompile Success");
			break;
	}
}

/****
 *	Cleanup
 *
 ****/

void	CBkgndTask::Cleanup(void)
{

	AnnounceWithSound();

	if (CompletionCode() == RC_INTERNAL_ERROR) {
		InternalError();
	}

	else if (CompletionCode() == RC_ERRORS) {
		CompilerError();
	}
	
	else if (CompletionCode() == RC_SUCCESS) {
		NoError();
	}

}

/****
 *	InternalError()
 *
 *	Process internal error information and present it to the user.
 *
 ****/

void	CBkgndTask::InternalError(void)
{
	Str255 str, errFile;
	FILE *fp;

	/* Get full name, change it to a C string
	 * and try to open the file
	 */
	gFileMgr->GetPartName(str);
	gFileMgr->GetErrFile(errFile);
	PtoCstr(errFile);
	fp = fopen((char *)errFile, "r");

	if (fp == NULL) {
		/* This should never happen, but just in case put
		 * up a dialogue box with some generic message.
		 */
		ParamText(str, "\p finished with RC_INTERNAL_ERROR. ",
			"\pError file with information about the internal ",
			"\perror couldn't be read.");
		gAdaApp->FinishedBkgnd(20000, false);	// false = CautionAlert
	}
	else {
		Str255 lines[3];
		int i;

		/* Read the first three lines from the error file
		 * (the file cannot be longer than 3 lines of 255
		 * characters).  The rest of the file is ignored.
		 */

		for (i = 0; i < 3; i++)
			lines[i][0] = 0;

		for (i = 0; (i < 3) && (!feof(fp)); i++) {
			fgets((char *)&lines[i], 255, fp);
			CtoPstr((char *)&lines[i]);
		}

		fclose(fp);
		// Don't remove the error file.  It serves as a bug report
		// to us (remove((char *)errFile);)

		ParamText(str, lines[0], lines[1], lines[2]);
		gAdaApp->FinishedBkgnd(20060, true);	// true = NoteAlert
	}
}

/****
 *	CompilerError
 *
 ****/

void	CBkgndTask::CompilerError(void)
{
}

/****
 *	NoError()
 *
 *	Processing when the compiler-part has finished and no errors were
 *	found.
 ****/

void	CBkgndTask::NoError(void)
{
}

